(*********************************************************************************
* Copyright: Bernecker+Rainer
* Author:    Mikael Regard
* Created:   May 13, 2019/8:21 AM 
* Updated:   Aug 19, 2019/09:00 AM
 *********************************************************************************
 * Description:
 * FuB to move Xbot in a synchronous linear motion.
 * Max 4 xbots can be moved at the same time.
 * 
 *Input:
 * MsgFrame     	(Profinet output frame)
 * ReadFrame    	(Profinet reply frame)
 * Execute      	(Edgepos, Execute = TRUE will run the FuB once)
 * Cmd Label   		(Optional 2 byte Command label. Used to identify specfic commands. 
 * Xbot ID[0..3] 	(Xbot to Move. XbotID > 0. Can only move 1 at the time. Virtual Axis ID 100-127, Macro ID 128-191
 * NumOfXbot    	(Number of Xbot to move)
 * TargetX[0..3]      ( Absolute Target X Position in mm)
 * TargetY[0..3]      ( Absolute Target Y Position in mm)
 * MaxSpeed[0..3]     (Maximum allowed speed. Will be automatically adjusted if they are below or above the configured system limit settings)
 * MaxAccel[0..3]     (Maximum allowed acceleration. Will be automatically adjusted if they are below or above the configured system limit settings)
 * FinalSpeed[0..3]   (Final speed = 0, Xbot will stop at the target position. Final speed > 0 can be used to connect multiple motion commands seamlessly.
 * 
 *
 * Output:
 * Error (Indicates if there is an error)
 * Busy  (FuB will be Busy until reply)
 * Done  (When Accepted command is received, Done = TRUE)
 * TravelTime (Longest expected travel time among the xbots) 
 * ErrorID (0x0001 = System Error, 0x2000 = Wrong PMC State, 0x2001 = No Mastership, 0x2002 = Mastership timeout,
 *		    0x2003 = Wrong group state, 0x2004 = Wrong Macro state, 0x2005 = Wrong Digital IO state,
 *          0x2006 = Wrong flyways state, 0x3000 = Wrong Xbot state, 0x4000 = Paramter Error: Invalid Parameters.
 *********************************************************************************)






FUNCTION_BLOCK PM_SyncLinearMotion		
	// Check Reply
	IF(Busy)THEN
		brsmemcpy(ADR(CmdLbl),ADR(ReadFrame[3]),2);
		IF(ReadFrame[0] = CmdCount AND
			ReadFrame[1] 	= 19  AND
			ReadFrame[2]	= 11 AND 
			CmdLbl = CmdLabel)THEN
			Done 	:= TRUE;
			Busy 	:= FALSE;
			brsmemcpy(ADR(ErrorID),ADR(ReadFrame[5]),2);
			Error 	:= UINT_TO_BOOL(ErrorID);
			Done 	:= NOT(UINT_TO_BOOL(ErrorID));
			brsmemcpy(ADR(TravelTime),ADR(ReadFrame[12]),4);
		ELSE
			Busy := FALSE;
		END_IF;
	END_IF;
	//Send Command		
	IF(Execute AND NOT(Done) AND NOT(Error))THEN
		IF(NOT(Busy))THEN
			brsmemset(ADR(MsgFrame[1]),FALSE,98);
			CmdCount 		:= ReadFrame[0] +1;
			MsgFrame[0] 	:= CmdCount; 
			MsgFrame[1] 	:= 19;
			MsgFrame[2] 	:= 11;
			brsmemcpy(ADR(MsgFrame[3]), ADR(CmdLabel), 2);
			MsgFrame[5]		:= 1;
			MsgFrame[9] 	:= NumOfXbot;
			
			FOR count := 0 TO NumOfXbot DO
				
				MsgFrame[11+(SYNC_BYTE_SEQUENCE*count)] := XbotID[count];
				Xmm2m[count] := TargetX[count]/1000.0;
				Ymm2m[count] := TargetY[count]/1000.0;
				
				brsmemcpy(ADR(MsgFrame[12+(SYNC_BYTE_SEQUENCE*count)]), ADR(Xmm2m[count]), 4);
				brsmemcpy(ADR(MsgFrame[16+(SYNC_BYTE_SEQUENCE*count)]), ADR(Ymm2m[count]), 4);
				brsmemcpy(ADR(MsgFrame[20+(SYNC_BYTE_SEQUENCE*count)]), ADR(FinalSpeed[count]), 4);
				brsmemcpy(ADR(MsgFrame[24+(SYNC_BYTE_SEQUENCE*count)]), ADR(MaxSpeed[count]), 4);
				brsmemcpy(ADR(MsgFrame[28+(SYNC_BYTE_SEQUENCE*count)]), ADR(MaxAccel[count]), 4);  
			END_FOR;
			
			Busy 			:= TRUE;	
		END_IF;
	//Reset output status	
	ELSIF(NOT(Execute))THEN
		Busy 	:= FALSE;
		Done 	:= FALSE;
		Error 	:= FALSE;
		ErrorID := 0;
	END_IF
	
	
	
END_FUNCTION_BLOCK
